using System;
using System.Net;
using System.Net.Sockets;
using System.Reflection;
using gov.va.med.vbecs.Common.Log;
using gov.va.med.VBECS.Communication.Clients;
using gov.va.med.VBECS.Communication.Common;
using gov.va.med.VBECS.Communication.Protocols;
using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;
using gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages;
using VistaLinkSequentialMessager = gov.va.med.VBECS.Communication.Channels.SequentialMessager<gov.va.med.VBECS.Communication.Clients.IClient, gov.va.med.vbecs.DAL.VistALink.Client.VistaLinkPinger>;

namespace gov.va.med.vbecs.DAL.VistALink.Client
{
	/// <summary>
	/// Used to notify VistALink consuming classes of connection availability change.
	/// </summary>
	public delegate void VistALinkConnectionStateChangedDelegate( object sender, VistALinkConnectionStateChangedEventArgs e );

	/// <summary>
	/// Class internally utilized by <see cref="RpcBroker"/> class to handle network operation.
	/// Maintains connection keeping it alive with heartbeats, sends and receives VistALink messages. 
	/// </summary>
#if DEBUG || UNIT_TEST
	public
#else
	internal 
#endif		
    class VistALinkClientConnection : IDisposable
	{
        // Messager that used queue messages
        // Wraps TCP/IP client 
        private readonly VistaLinkSequentialMessager _messager;
        /// <summary>
        /// Connection state change event.
        /// </summary>
        public event VistALinkConnectionStateChangedDelegate ConnectionStateChanged;

        private readonly VistALinkFaultExceptionFactory _faultXcpFactory = new VistALinkFaultExceptionFactory();

        // Logger object
	    readonly ILogger _logger = LogManager.Instance().LoggerLocator.GetLogger(MethodBase.GetCurrentMethod().DeclaringType);

        #region Private Methods

	    private IClient Client
	    {
            get { return _messager == null ? null : _messager.Messager; }
	    }

	    /// <summary>
        /// Disconnects from VistA server with or without prior 
        /// notification (depending on parameter specified).
        /// </summary>
        /// <param name="notifyServer">
        ///		Flag indicating whether server should be notified before disconnect.
        ///		Using 'false' value will immediately drop communication with the server.
        ///	</param>
        private void Disconnect(bool notifyServer)
        {
            _logger.Debug(string.Format("Disconnects from VistA server {0} prior notification", notifyServer ? "with" : "without"));

	        if (notifyServer)
	        {
                // the following call closes connection (closes Tcp socket)
	            notify_vista_about_disconnect();
	        }

            Client.Disconnect();
            _messager.Stop();
        }

        private void notify_vista_about_disconnect()
        {
            VistALinkCloseSocketResponseMessage response;
            try
            {
                // If there are any queued messages, disconnect message is queued at the end the following call block until disconnect message is executed
                response = (VistALinkCloseSocketResponseMessage)_messager.SendReceive(new VistALinkCloseSocketRequestMessage(), GlobalConfig.DefaultRpcRequestTimeout.ToInt32Milliseconds());
            }
            catch (Exception ex)
            {
                throw (new VistALinkNetworkErrorException(SR.Exceptions.GeneralNetworkError(), ex));
            }
            if (response.ResponseStatus != ResponseStatus.Success)
                throw (new VistALinkException(SR.Exceptions.ServerFailureWhileClosingConnection()));
        }

	    #endregion

        /// <summary>
        /// Initializes the object, and opens a connection to VistA server. 
        /// </summary>
        /// <param name="serverConnectionInfo">Remote M server connection info.</param>
        public VistALinkClientConnection(ServerConnectionInfo serverConnectionInfo)
        {
            if (serverConnectionInfo == null)
                throw (new ArgumentNullException("serverConnectionInfo"));
            ServerConnectionInfo = serverConnectionInfo;

            //_theTimeoutMilliseconds = (int)GlobalContext.Instance().AppSettingsReader.GetValue("VistAClientTimeout", typeof(int));

            _messager = new VistaLinkSequentialMessager(ClientFactory.CreateClient<MockPinger>(
                new IPEndPoint(serverConnectionInfo.IPAddress, serverConnectionInfo.PortNumber), 
                new MessagesProtocolEot(null, new VistaLinkMessageFactory())));
        }

	    /// <summary>
	    /// Initializes the object, and opens a connection to VistA server. 
	    /// </summary>
	    /// <param name="serverConnectionInfo">Remote M server connection info.</param>
	    /// <param name="clinet">Underlying client</param>
	    public VistALinkClientConnection(ServerConnectionInfo serverConnectionInfo, IClient clinet)
        {
            if (serverConnectionInfo == null)
                throw (new ArgumentNullException("serverConnectionInfo"));

            ServerConnectionInfo = serverConnectionInfo;

            _messager = new VistaLinkSequentialMessager(clinet);
        }

        /// <summary>
        /// Connects to VistA server.
        /// </summary>
        public void Connect()
        {
            if (Client.CommunicationStatus == CommunicationStatus.Connected)
                throw (new InvalidOperationException(SR.Exceptions.VistALinkConnectionIsAlreadyOpen()));

            try
            {
                _logger.Debug(string.Format("Start connect to VistA server: IP={0}, PORT={1}",
                    ServerConnectionInfo.IPAddress, ServerConnectionInfo.PortNumber));
                Client.Connect();
            }
            catch (Exception e)
            {
                _logger.Error("Can't connect to VistA server", e);
                if (e is SocketException || e is TimeoutException)
                {
                    _logger.Debug("SocketException and TimeoutException are wrapped to VistALinkNetworkErrorException and re-throwed");
                    throw (new VistALinkNetworkErrorException(SR.Exceptions.GeneralNetworkError(), e));
                }
            }
            _logger.Debug("Successfully connected to VistA server");

            Client.Disconnected += client_on_disconnected;
            _messager.Start();

            // Fire connection state is connected event
            OnConnectionStateChanged(new VistALinkConnectionStateChangedEventArgs(true));
        }

	    private void client_on_disconnected(object sender, EventArgs eventArgs)
	    {
            _logger.Debug("client_on_disconnected called, inform subscribers");
            OnConnectionStateChanged(new VistALinkConnectionStateChangedEventArgs(false));
	    }

	    /// <summary>
        /// Indicates whether connection with VistA server is established. 
        /// </summary>
        public bool IsConnected
        {
            get { return Client.CommunicationStatus == CommunicationStatus.Connected; }
        }

	    /// <summary>
	    /// Remote server connection information.
	    /// </summary>
	    // TODO: remove it from here?
	    public ServerConnectionInfo ServerConnectionInfo { get; private set; }

	    /// <summary>
        /// Sends VistALink message to server and receives response message. 
        /// </summary>
        /// <param name="clientMessage">Message to send.</param>
        /// <returns>Server response message.</returns>
        public VistALinkMessage SendReceiveMessage(VistALinkMessage clientMessage)
        {
            VistALinkMessage response;
            try
            {
                _logger.Debug("Start SendReceive a message");
                // If there are any queued messages, a message is queued at the end the following call block until a message is executed
                // Including disconnect message
                response = (VistALinkMessage) _messager.SendReceive(clientMessage, 
                    clientMessage is RpcRequestMessage ? ((RpcRequestMessage)clientMessage).RpcRequest.RpcTimeout.ToInt32Milliseconds() : GlobalConfig.DefaultRpcRequestTimeout.ToInt32Milliseconds());

                var message = response as BaseFaultMessage;
                if (message != null)
                {
                    _logger.Debug(string.Format("SendReceiveMessage: BaseFaultMessage received: {0}{1}Exception is re-throwed in VistALinkFaultException.", message.FaultInfo, Environment.NewLine));
                    throw (_faultXcpFactory.CreateVistALinkExceptionFromFaultMessage(message));
                }
            }
            catch (VistALinkFaultException e)
            {
                if (e.IsFatal)
                {
                    _logger.Error("SendReceiveMessage: Fatal VistALinkFaultException received, disconnect from Vista without notification");
                    Disconnect(false);
                }

                _logger.Debug("SendReceiveMessage: VistALinkFaultException received and re-throwed");
                throw;
            }
            catch(Exception e)
            {
                _logger.Error("SendReceiveMessage: exception is received, disconnect from Vista without notification and re-throw currect exception as VistALinkFaultException", e);
                Disconnect(false);
                //CR3414: wrap as VistALinkNetworkErrorException (expected by outer functionality)
                throw new VistALinkNetworkErrorException("VistALink: failed to Send/Receive message. Error: " + e.Message, e);
            }

            _logger.Debug("SendReceiveMessage: Message successfully received");
	        if (!(clientMessage is VistALinkCloseSocketRequestMessage)) return response;

	        _logger.Debug("SendReceiveMessage: request message was VistALinkCloseSocketRequestMessage, so disconnect from Vista without notification.");
	        Disconnect(false);

	        return response;
        }

        /// <summary>
        /// Connection state change notification method.
        /// </summary>
        /// <param name="eventArgs">Connection state change event args.</param>
        protected virtual void OnConnectionStateChanged(VistALinkConnectionStateChangedEventArgs eventArgs)
        {
            if (ConnectionStateChanged != null)
                ConnectionStateChanged.Invoke(this, eventArgs);
        }


        #region Dispose

        /// <summary>
        /// Finalization method closing the connection (disconnecting from server). 
        /// </summary>
        ~VistALinkClientConnection()
        {
            Dispose(false);
        }

        /// <summary>
        /// Cleanup method needed to implement <see cref="IDisposable"/>.
        /// </summary>
        public void Dispose()
        {
            _logger.Debug("Dispose");
            GC.SuppressFinalize(this);
            Dispose(true);
        }

        /// <summary>
        /// Closes the connection.
        /// </summary>
        public void Close()
        {
            _logger.Debug("Close");
            Dispose();
        }

        /// <summary>
        /// Standard implementation of the dispose method. 
        /// </summary>
        /// <param name="disposing">
        ///		Flag indicating if the disposition was invoked explicitly under normal 
        ///		conditions (true) or forced upon object disposal (false).
        ///	</param>
        private void Dispose(bool disposing)
        {
            _logger.Debug("Dispose, disposing=" + disposing);
            lock (this)
            {
                if (disposing && Client.CommunicationStatus == CommunicationStatus.Connected)
                    Disconnect(true);

                _logger.Debug("Disconnect underlying client and messages.");
                Client.Dispose();
                _messager.Dispose();
            }
        }


        #endregion
	}
}
